import React, { Component } from 'react';
import fillTemplate from 'es6-dynamic-template';
import { History } from 'history';
import { withStyles } from '@material-ui/core/styles';
import InfoTooltip from 'components/InfoTooltip';
import { Logger } from 'library/Logger';
import StyledButton from 'components/StyledButton';
import routes from 'constants/routes';
import Select from '@material-ui/core/Select';
import cn from 'classnames';
import MenuItem from '@material-ui/core/MenuItem';
import styles from './styles';
import { QCC_FW_LANGUAGES } from '../../../resources/strings/languages';

type Props = {
    device: object,
    strings: object,
    history: History,
    language: object,
    hasBetaVersion: boolean,
    betaVersion: string,
    onReInstallFirmware: language => {},
    classes: {
        [key: string]: string
    }
};

class DeviceUpToDateScreen extends Component<Props> {
    constructor(props) {
        super(props);
        this.state = {
            device: props.device,
            language: props.language,
            fwLanguage: null
        };
    }

    static getDerivedStateFromProps(nextProps, prevState) {
        const { language, device } = prevState;
        if (language !== nextProps.language) {
            return {
                language: nextProps.language,
                strings: nextProps.strings
            };
        }
        if (device.connected && !nextProps.device.connected) {
            // Device disconnected
            this.moveToMainScreen();
        }
        return null;
    }

    componentDidMount() {
        Logger.info('UnsupportedUnitScreen: Mounted');
    }

    moveToMainScreen() {
        const { history } = this.props;
        setTimeout(() => {
            // Change screen (but not within the rendering functions)
            // Move to main screen, but do not perform auto update checks again
            history.replace({
                pathname: routes.WELCOME.path,
                state: { skipAutoUpdate: true }
            });
        }, 100);
    }

    renderFWLanguage() {
        const { device, classes } = this.props;
        const { fwLanguage } = this.state;
        const language = fwLanguage || device.deviceLanguage;

        return (
            <Select
                className={cn(classes.language, classes.languageSelect)}
                value={language}
                onChange={event =>
                    this.setState({ fwLanguage: event.target.value })
                }
                MenuProps={{
                    classes: {
                        paper: classes.languagePaper
                    }
                }}
                disableUnderline
                inputProps={{
                    classes: {
                        icon: classes.languageInputArrow,
                        root: classes.languageInput,
                        select: classes.languageSelect
                    }
                }}
            >
                {Object.keys(QCC_FW_LANGUAGES)
                    .filter(key =>
                        QCC_FW_LANGUAGES[key].models.includes(device.model)
                    )
                    .map(key => (
                        <MenuItem
                            className={cn(
                                classes.languageMenuItem,
                                key === language
                                    ? classes.selectedLanguageItem
                                    : ''
                            )}
                            key={key}
                            value={key}
                        >
                            <img alt={key} src={QCC_FW_LANGUAGES[key].icon} />{' '}
                            {QCC_FW_LANGUAGES[key].title}
                        </MenuItem>
                    ))}
            </Select>
        );
    }

    render() {
        const {
            onReInstallFirmware,
            strings,
            device,
            hasBetaVersion,
            betaVersion,
            classes
        } = this.props;
        const { fwLanguage } = this.state;
        const isBetaInstalled = !!(
            betaVersion && betaVersion.toString() === device.fwVersion
        );
        const filteredLanguages = Object.keys(QCC_FW_LANGUAGES).filter(key =>
            QCC_FW_LANGUAGES[key].models.includes(device.model)
        );

        return (
            <div className={classes.container}>
                <div className={classes.currentVersion}>
                    {fillTemplate(strings.device_firmware_version_is, {
                        version: device.fwVersion,
                        betaBuildNumber: `${
                            hasBetaVersion &&
                            isBetaInstalled &&
                            device.btmVersion
                                ? `(${device.btmVersion})`
                                : ''
                        }`
                    })}
                </div>
                <div className={classes.title}>
                    {strings.device_up_to_date_title}
                </div>
                {device.deviceLanguage && filteredLanguages.length > 0 && (
                    <div className={classes.languageContainer}>
                        <div className={classes.fwLanguageTitle}>
                            {strings.fw_language}
                        </div>
                        {this.renderFWLanguage()}
                    </div>
                )}
                <div
                    className={cn(
                        classes.btnContainer,
                        device.deviceLanguage
                            ? classes.btnContainerNoMargin
                            : null
                    )}
                >
                    <StyledButton
                        styles={classes.reinstallButton}
                        btnSize="big"
                        onClick={() =>
                            onReInstallFirmware(
                                fwLanguage || device.deviceLanguage
                            )
                        }
                    >
                        {strings.device_up_to_date_reinstall_firmware_button}
                    </StyledButton>
                    <InfoTooltip
                        styles={{ popper: classes.tooltipPopper }}
                        placement="right-start"
                        iconClasses={classes.infoIcon}
                        iconType="white"
                    >
                        <div className={classes.tooltipColumn}>
                            <div className={classes.tooltipLabel}>
                                {
                                    strings.device_up_to_date_reinstall_firmware_popup
                                }
                            </div>
                        </div>
                    </InfoTooltip>
                </div>
            </div>
        );
    }
}

export default withStyles(styles)(DeviceUpToDateScreen);
